import express from 'express';
import { User, File, ActivityLog, Notification } from '../models/index.js';
import { authenticate, authorize } from '../middleware/auth.js';
import { validateUserUpdate, validateUUID } from '../middleware/validation.js';
import { logUserActivity } from '../middleware/logging.js';
import { AppError, NotFoundError } from '../middleware/errorHandler.js';
import { logger } from '../utils/logger.js';

const router = express.Router();

// All routes require authentication
router.use(authenticate);

// Get current user profile
router.get('/profile', async (req, res, next) => {
  try {
    const user = await User.findByPk(req.user.id, {
      include: [
        {
          model: File,
          as: 'files',
          where: { is_deleted: false },
          required: false,
          attributes: ['id', 'type', 'size']
        }
      ]
    });

    if (!user) {
      throw new NotFoundError('User');
    }

    // Calculate storage statistics
    const files = user.files || [];
    const stats = {
      totalFiles: files.filter(f => f.type === 'file').length,
      totalFolders: files.filter(f => f.type === 'folder').length,
      storageUsed: user.storage_used,
      storageQuota: user.storage_quota,
      storagePercentage: (user.storage_used / user.storage_quota) * 100
    };

    res.json({
      user: user.toJSON(),
      stats
    });
  } catch (error) {
    next(error);
  }
});

// Update user profile
router.patch('/profile', validateUserUpdate, logUserActivity('update_profile'), async (req, res, next) => {
  try {
    const { name, preferences } = req.body;
    
    const user = await User.findByPk(req.user.id);
    if (!user) {
      throw new NotFoundError('User');
    }

    // Update allowed fields
    if (name) user.name = name;
    if (preferences) user.preferences = { ...user.preferences, ...preferences };

    await user.save();

    logger.info(`User profile updated: ${user.email}`);

    res.json({
      message: 'Profile updated successfully',
      user: user.toJSON()
    });
  } catch (error) {
    next(error);
  }
});

// Get user activity logs
router.get('/activity', async (req, res, next) => {
  try {
    const { limit = 50, offset = 0 } = req.query;

    const activities = await ActivityLog.findAll({
      where: { user_id: req.user.id },
      order: [['created_at', 'DESC']],
      limit: parseInt(limit),
      offset: parseInt(offset),
      include: [
        {
          model: User,
          as: 'user',
          attributes: ['id', 'name', 'email']
        }
      ]
    });

    const total = await ActivityLog.count({
      where: { user_id: req.user.id }
    });

    res.json({
      activities,
      pagination: {
        total,
        limit: parseInt(limit),
        offset: parseInt(offset),
        hasMore: (parseInt(offset) + parseInt(limit)) < total
      }
    });
  } catch (error) {
    next(error);
  }
});

// Get user notifications
router.get('/notifications', async (req, res, next) => {
  try {
    const { includeRead = false, limit = 50 } = req.query;

    const notifications = await Notification.findByUser(
      req.user.id, 
      includeRead === 'true'
    );

    const unreadCount = await Notification.getUnreadCount(req.user.id);

    res.json({
      notifications: notifications.slice(0, parseInt(limit)),
      unreadCount
    });
  } catch (error) {
    next(error);
  }
});

// Mark notification as read
router.patch('/notifications/:id/read', validateUUID('id'), async (req, res, next) => {
  try {
    const notification = await Notification.findOne({
      where: {
        id: req.params.id,
        user_id: req.user.id
      }
    });

    if (!notification) {
      throw new NotFoundError('Notification');
    }

    await notification.markAsRead();

    res.json({
      message: 'Notification marked as read'
    });
  } catch (error) {
    next(error);
  }
});

// Mark all notifications as read
router.patch('/notifications/read-all', async (req, res, next) => {
  try {
    await Notification.markAllAsRead(req.user.id);

    res.json({
      message: 'All notifications marked as read'
    });
  } catch (error) {
    next(error);
  }
});

// Get user sessions
router.get('/sessions', async (req, res, next) => {
  try {
    const sessions = await UserSession.findByUser(req.user.id);

    const sessionsWithCurrent = sessions.map(session => ({
      ...session.toJSON(),
      isCurrent: session.token === req.session.token
    }));

    res.json({ sessions: sessionsWithCurrent });
  } catch (error) {
    next(error);
  }
});

// Revoke user session
router.delete('/sessions/:id', validateUUID('id'), async (req, res, next) => {
  try {
    const session = await UserSession.findOne({
      where: {
        id: req.params.id,
        user_id: req.user.id
      }
    });

    if (!session) {
      throw new NotFoundError('Session');
    }

    await session.revoke();

    res.json({
      message: 'Session revoked successfully'
    });
  } catch (error) {
    next(error);
  }
});

// Revoke all other sessions
router.delete('/sessions', async (req, res, next) => {
  try {
    await UserSession.revokeAllForUser(req.user.id, req.session.token);

    res.json({
      message: 'All other sessions revoked successfully'
    });
  } catch (error) {
    next(error);
  }
});

// Change password
router.patch('/password', async (req, res, next) => {
  try {
    const { currentPassword, newPassword } = req.body;

    if (!currentPassword || !newPassword) {
      throw new AppError('Current password and new password are required', 400);
    }

    if (newPassword.length < 6) {
      throw new AppError('New password must be at least 6 characters long', 400);
    }

    const user = await User.findOne({
      where: { id: req.user.id },
      attributes: { include: ['password'] }
    });

    if (!user) {
      throw new NotFoundError('User');
    }

    // Verify current password
    const isCurrentPasswordValid = await user.comparePassword(currentPassword);
    if (!isCurrentPasswordValid) {
      throw new AppError('Current password is incorrect', 400);
    }

    // Update password
    user.password = newPassword;
    await user.save();

    // Revoke all other sessions for security
    await UserSession.revokeAllForUser(req.user.id, req.session.token);

    logger.info(`Password changed for user: ${user.email}`);

    res.json({
      message: 'Password changed successfully'
    });
  } catch (error) {
    next(error);
  }
});

// Delete user account
router.delete('/account', async (req, res, next) => {
  try {
    const { password } = req.body;

    if (!password) {
      throw new AppError('Password is required to delete account', 400);
    }

    const user = await User.findOne({
      where: { id: req.user.id },
      attributes: { include: ['password'] }
    });

    if (!user) {
      throw new NotFoundError('User');
    }

    // Verify password
    const isPasswordValid = await user.comparePassword(password);
    if (!isPasswordValid) {
      throw new AppError('Password is incorrect', 400);
    }

    // Soft delete user (deactivate)
    user.is_active = false;
    await user.save();

    // Revoke all sessions
    await UserSession.revokeAllForUser(req.user.id);

    logger.info(`User account deleted: ${user.email}`);

    res.json({
      message: 'Account deleted successfully'
    });
  } catch (error) {
    next(error);
  }
});

// Get current user's storage usage
router.get('/me/storage', async (req, res, next) => {
  try {
    const userId = req.user.id;
    // Sum file sizes for all files owned by the user (not deleted)
    const result = await File.findAll({
      where: { owner_id: userId, is_deleted: false, type: 'file' },
      attributes: [[File.sequelize.fn('SUM', File.sequelize.col('size')), 'used']],
      raw: true
    });
    const used = Number(result[0].used) || 0;
    // For demo, set total to 10GB (10240 MB)
    const total = 10240 * 1024 * 1024;
    res.json({ used: Math.round(used / (1024 * 1024)), total: Math.round(total / (1024 * 1024)) }); // MB
  } catch (error) {
    next(error);
  }
});

// Update current user's profile
router.patch('/me', async (req, res, next) => {
  try {
    const user = await User.findByPk(req.user.id);
    if (!user) return res.status(404).json({ error: 'User not found' });
    const { name, email, password } = req.body;
    if (name) user.name = name;
    if (email) user.email = email;
    if (password) user.password = password; // Assume password hashing in model hook
    await user.save();
    res.json({ user });
  } catch (error) {
    next(error);
  }
});

// Debug endpoint: List all files for the current user (ignoring parent/folder)
router.get('/debug/files', async (req, res, next) => {
  try {
    const files = await File.findAll({
      where: { owner_id: req.user.id },
      order: [['createdAt', 'DESC']]
    });
    res.json({ files });
  } catch (error) {
    next(error);
  }
});

export default router;