import express from 'express';
import { File, FileVersion, User, Trash, ShareLink, FileShare } from '../models/index.js';
import { authenticate, authorize } from '../middleware/auth.js';
import { 
  validateFileUpload, 
  validateFolderCreate, 
  validateFileMove,
  validateUUID,
  validateSearch
} from '../middleware/validation.js';
import { logFileActivity } from '../middleware/logging.js';
import { s3Service } from '../utils/s3.js';
import { AppError, NotFoundError } from '../middleware/errorHandler.js';
import { logger } from '../utils/logger.js';
import { v4 as uuidv4 } from 'uuid';
import { PutObjectCommand, ListObjectsV2Command } from '@aws-sdk/client-s3';
import { Op } from 'sequelize';
import archiver from 'archiver';

const router = express.Router();

// All routes require authentication
router.use(authenticate);

// List trash for current user (must be before any /:id route)
router.get('/trash', async (req, res, next) => {
  try {
    const trashItems = await Trash.findAll({
      where: { owner_id: req.user.id },
      include: [{ model: File, as: 'file' }],
      order: [['deleted_at', 'DESC']]
    });
    res.json({ trash: trashItems });
  } catch (error) {
    next(error);
  }
});

// Get files and folders
router.get('/', async (req, res, next) => {
  try {
    const { parentId, type, starred, shared, deleted } = req.query;
    const where = {};

    // If requesting root (top-level) folders, show all user root folders
    if (!parentId || parentId === 'root') {
      where.parent_id = null;
      where.is_deleted = false;
    } else {
      where.parent_id = parentId;
      if (deleted !== undefined) {
        where.is_deleted = deleted === 'true';
      } else {
        where.is_deleted = false;
      }
    }

    if (type && type !== 'all') {
      where.type = type;
    }
    if (starred === 'true') {
      where.is_starred = true;
    }
    if (shared === 'true') {
      where.is_shared = true;
    }

    // Admins and all users can see all files (no owner or sharing filter)
    // (If you want to restrict, add back the previous logic)

    const files = await File.findAll({
      where,
      order: [['type', 'DESC'], ['name', 'ASC']],
      include: [
        {
          model: User,
          as: 'owner',
          attributes: ['id', 'name', 'email']
        }
      ]
    });

    // Map mime_type to mimeType for frontend compatibility
    const filesWithMimeType = files.map(f => {
      const file = f.toJSON();
      file.mimeType = file.mime_type;
      return file;
    });

    res.json({ files: filesWithMimeType });
  } catch (error) {
    next(error);
  }
});

// Search files
router.get('/search', validateSearch, async (req, res, next) => {
  try {
    const { q: query, type = 'all', limit = 50 } = req.query;
    
    const where = {
      owner_id: req.user.id,
      is_deleted: false,
      name: {
        [File.sequelize.Sequelize.Op.iLike]: `%${query}%`
      }
    };

    if (type !== 'all') {
      where.type = type;
    }

    const files = await File.findAll({
      where,
      limit: parseInt(limit),
      order: [['updated_at', 'DESC']],
      include: [
        {
          model: User,
          as: 'owner',
          attributes: ['id', 'name', 'email']
        }
      ]
    });

    res.json({ files, query });
  } catch (error) {
    next(error);
  }
});

// Global search: all files by name, regardless of user
router.get('/global-search', async (req, res, next) => {
  try {
    const { q = '', limit = 50 } = req.query;
    const files = await File.findAll({
      where: {
        is_deleted: false,
        name: { [File.sequelize.Sequelize.Op.iLike]: `%${q}%` }
      },
      order: [['updatedAt', 'DESC']],
      limit: parseInt(limit, 10) || 50,
      include: [
        { model: User, as: 'owner', attributes: ['id', 'name', 'email'] }
      ]
    });
    res.json({ files });
  } catch (error) {
    next(error);
  }
});

// Get file details
router.get('/:id', validateUUID('id'), async (req, res, next) => {
  try {
    const file = await File.findOne({
      where: {
        id: req.params.id,
        owner_id: req.user.id
      },
      include: [
        {
          model: User,
          as: 'owner',
          attributes: ['id', 'name', 'email']
        },
        {
          model: FileVersion,
          as: 'versions',
          order: [['version', 'DESC']],
          limit: 10
        }
      ]
    });

    if (!file) {
      throw new NotFoundError('File');
    }

    res.json({ file });
  } catch (error) {
    next(error);
  }
});

// Create folder
router.post('/folders', validateFolderCreate, logFileActivity('create_folder'), async (req, res, next) => {
  try {
    const { name, parentId } = req.body;

    // Check if parent folder exists and belongs to user
    let parentFolder = null;
    if (parentId) {
      parentFolder = await File.findOne({
        where: {
          id: parentId,
          owner_id: req.user.id,
          type: 'folder',
          is_deleted: false
        }
      });
      if (!parentFolder) {
        throw new NotFoundError('Parent folder');
      }
    }

    // Check if folder with same name already exists in the same location
    const existingFolder = await File.findOne({
      where: {
        name,
        parent_id: parentId || null,
        owner_id: req.user.id,
        type: 'folder',
        is_deleted: false
      }
    });
    if (existingFolder) {
      throw new AppError('Folder with this name already exists', 409);
    }

    // Generate path
    let path = `/${name}`;
    let s3FolderPath = `${name}/`;
    if (parentFolder) {
      path = `${parentFolder.path}/${name}`;
      s3FolderPath = `${parentFolder.path.replace(/^\//, '')}/${name}/`;
    }

    // Create folder in DB
    const folder = await File.create({
      name,
      type: 'folder',
      path,
      parent_id: parentId || null,
      owner_id: req.user.id
    });

    // Create folder in S3 (zero-byte object with trailing slash)
    const userEmail = req.user.email;
    const s3Key = s3Service.generateFolderKey(userEmail, s3FolderPath);
    await s3Service.client.send(new PutObjectCommand({
      Bucket: s3Service.bucketName,
      Key: s3Key,
      Body: ''
    }));

    logger.info(`Folder created: ${folder.name} by user ${req.user.email} (S3: ${s3Key})`);

    res.status(201).json({
      message: 'Folder created successfully',
      folder
    });
  } catch (error) {
    next(error);
  }
});

// Get upload URL for file
router.post('/upload-url', validateFileUpload, async (req, res, next) => {
  // Debug: log incoming body and types
  console.log('DEBUG /upload-url req.body:', req.body);
  if (req.body) {
    Object.entries(req.body).forEach(([k, v]) => {
      console.log(`  ${k}:`, v, 'type:', typeof v);
    });
  }
  try {
    const { name, size, mimeType, parentId } = req.body;

    // Validate file type and size
    if (!s3Service.isAllowedFileType(mimeType)) {
      throw new AppError('File type not allowed', 400);
    }

    if (!s3Service.isAllowedFileSize(size)) {
      throw new AppError('File size exceeds limit', 413);
    }

    // Check if parent folder exists
    if (parentId !== undefined && parentId !== null) {
      const parentFolder = await File.findOne({
        where: {
          id: parentId,
          owner_id: req.user.id,
          type: 'folder',
          is_deleted: false
        }
      });
      if (!parentFolder) {
        return res.status(404).json({ error: 'Parent folder not found', parentId });
      }
    }

    // Generate file ID and S3 key
    const fileId = uuidv4();
    const userEmail = req.user.email;
    const s3Key = s3Service.generateFileKey(userEmail, fileId, name);

    // Generate presigned upload URL
    const uploadUrl = await s3Service.getUploadUrl(s3Key, mimeType);

    // Generate path
    let path = `/${name}`;
    if (parentId) {
      const parentFolder = await File.findByPk(parentId);
      path = `${parentFolder.path}/${name}`;
    }

    // Create file record (pending upload)
    const file = await File.create({
      id: fileId,
      name,
      type: 'file',
      size,
      mime_type: mimeType,
      path,
      s3_key: s3Key,
      parent_id: parentId || null,
      owner_id: req.user.id,
      metadata: {
        uploadStatus: 'pending'
      }
    });

    res.json({
      uploadUrl,
      fileId,
      file
    });
  } catch (error) {
    next(error);
  }
});

// Confirm file upload
router.post('/:id/confirm-upload', validateUUID('id'), logFileActivity('upload_file'), async (req, res, next) => {
  try {
    const file = await File.findOne({
      where: {
        id: req.params.id,
        owner_id: req.user.id
      }
    });

    if (!file) {
      throw new NotFoundError('File');
    }

    // Verify file exists in S3
    const exists = await s3Service.objectExists(file.s3_key);
    if (!exists) {
      throw new AppError('File upload not completed', 400);
    }

    // Get file metadata from S3
    const metadata = await s3Service.getObjectMetadata(file.s3_key);

    // Update file record
    file.size = metadata.size;
    file.s3_version_id = metadata.versionId;
    file.checksum = metadata.etag?.replace(/"/g, '');
    file.metadata = {
      ...file.metadata,
      uploadStatus: 'completed',
      uploadedAt: new Date().toISOString()
    };
    await file.save();

    // Update user storage usage
    await req.user.updateStorageUsed(metadata.size);

    // Create initial file version
    await FileVersion.create({
      file_id: file.id,
      version: 1,
      size: metadata.size,
      s3_key: file.s3_key,
      s3_version_id: metadata.versionId,
      checksum: file.checksum,
      mime_type: file.mime_type,
      is_latest: true,
      created_by: req.user.id
    });

    logger.info(`File uploaded: ${file.name} (${metadata.size} bytes) by user ${req.user.email}`);

    res.json({
      message: 'File uploaded successfully',
      file
    });
  } catch (error) {
    next(error);
  }
});

// Get download URL for file
router.get('/:id/download', validateUUID('id'), logFileActivity('download_file'), async (req, res, next) => {
  try {
    const file = await File.findOne({
      where: {
        id: req.params.id,
        owner_id: req.user.id,
        type: 'file',
        is_deleted: false
      }
    });

    if (!file) {
      throw new NotFoundError('File');
    }

    // Generate presigned download URL
    const downloadUrl = await s3Service.getDownloadUrl(file.s3_key);

    res.json({
      downloadUrl,
      fileName: file.name,
      size: file.size,
      mimeType: file.mime_type
    });
  } catch (error) {
    next(error);
  }
});

// Direct file stream for inline preview (images, pdf, text)
router.get('/:id/preview', validateUUID('id'), async (req, res, next) => {
  try {
    // Allow preview if user is owner, admin, or has share permissions
    let file = await File.findOne({
      where: {
        id: req.params.id,
        is_deleted: false
      }
    });
    if (!file) throw new NotFoundError('File');
    // Check access: owner, admin, or shared
    const isOwner = file.owner_id === req.user.id;
    const isAdmin = req.user.role === 'admin';
    let isShared = false;
    if (!isOwner && !isAdmin) {
      const share = await FileShare.findOne({ where: { file_id: file.id, user_id: req.user.id } });
      isShared = !!share;
    }
    if (!isOwner && !isAdmin && !isShared) {
      return res.status(403).json({ error: 'Not authorized to preview this file' });
    }
    const s3Stream = await s3Service.getObjectStream(file.s3_key);
    // Set headers for inline preview
    res.setHeader('Content-Type', file.mime_type || 'application/octet-stream');
    res.setHeader('Content-Disposition', 'inline; filename="' + encodeURIComponent(file.name) + '"');
    s3Stream.pipe(res);
  } catch (error) {
    next(error);
  }
});

// Move/rename file or folder
router.patch('/:id/move', validateUUID('id'), validateFileMove, logFileActivity('move_file'), async (req, res, next) => {
  try {
    const { targetParentId, newName } = req.body;

    const file = await File.findOne({
      where: {
        id: req.params.id,
        owner_id: req.user.id,
        is_deleted: false
      }
    });

    if (!file) {
      throw new NotFoundError('File');
    }

    // Check if target parent exists
    if (targetParentId) {
      const targetParent = await File.findOne({
        where: {
          id: targetParentId,
          owner_id: req.user.id,
          type: 'folder',
          is_deleted: false
        }
      });

      if (!targetParent) {
        throw new NotFoundError('Target folder');
      }

      // Prevent moving folder into itself or its children
      if (file.type === 'folder') {
        // This would require a recursive check in a real implementation
        if (targetParentId === file.id) {
          throw new AppError('Cannot move folder into itself', 400);
        }
      }
    }

    const oldPath = file.path;
    const oldName = file.name;

    // Update file/folder
    if (newName && newName !== file.name) {
      file.name = newName;
    }

    if (targetParentId !== undefined) {
      file.parent_id = targetParentId;
    }

    // Update path
    let newPath = `/${file.name}`;
    if (file.parent_id) {
      const parentFolder = await File.findByPk(file.parent_id);
      newPath = `${parentFolder.path}/${file.name}`;
    }
    file.path = newPath;

    await file.save();

    // If it's a file and was renamed, copy to new S3 key
    if (file.type === 'file' && newName && newName !== oldName) {
      const newS3Key = s3Service.generateFileKey(req.user.id, file.id, file.name);
      await s3Service.copyObject(file.s3_key, newS3Key);
      await s3Service.deleteObject(file.s3_key);
      file.s3_key = newS3Key;
      await file.save();
    }

    logger.info(`File moved: ${oldPath} -> ${newPath} by user ${req.user.email}`);

    res.json({
      message: 'File moved successfully',
      file
    });
  } catch (error) {
    next(error);
  }
});

// Star a file
router.post('/:id/star', async (req, res, next) => {
  try {
    const file = await File.findByPk(req.params.id);
    if (!file) return res.status(404).json({ error: 'File not found' });
    file.is_starred = true;
    await file.save();
    res.json({ message: 'File starred', file });
  } catch (error) {
    next(error);
  }
});

// Unstar a file
router.post('/:id/unstar', async (req, res, next) => {
  try {
    const file = await File.findByPk(req.params.id);
    if (!file) return res.status(404).json({ error: 'File not found' });
    file.is_starred = false;
    await file.save();
    res.json({ message: 'File unstarred', file });
  } catch (error) {
    next(error);
  }
});

// Get all starred files
router.get('/starred', async (req, res, next) => {
  try {
    const starredFiles = await File.findAll({ where: { is_starred: true } });
    res.json({ files: starredFiles });
  } catch (error) {
    next(error);
  }
});

// Move file to trash (soft delete)
router.delete('/:id', validateUUID('id'), logFileActivity('delete_file'), async (req, res, next) => {
  const { validationResult } = await import('express-validator');
  const errors = validationResult(req);
  if (!errors.isEmpty()) {
    console.error('Validation errors:', errors.array());
  }
  try {
    const file = await File.findOne({
      where: {
        id: req.params.id,
        owner_id: req.user.id,
        is_deleted: false
      }
    });

    if (!file) {
      throw new NotFoundError('File');
    }

    // Prevent deletion of user root folder
    if (file.parent_id === null && file.metadata && file.metadata.root) {
      return res.status(400).json({ error: 'Cannot delete user root folder.' });
    }

    // Mark as deleted (soft delete)
    file.is_deleted = true;
    file.deleted_at = new Date();
    await file.save();

    // Add to Trash table
    await Trash.create({ file_id: file.id, owner_id: req.user.id });

    logger.info(`File moved to trash: ${file.name} by user ${req.user.email}`);

    res.json({
      message: 'File moved to trash'
    });
  } catch (error) {
    next(error);
  }
});

// Restore file from trash
router.post('/:id/restore', validateUUID('id'), logFileActivity('restore_file'), async (req, res, next) => {
  try {
    const file = await File.findOne({
      where: {
        id: req.params.id,
        owner_id: req.user.id,
        is_deleted: true
      }
    });
    if (!file) {
      throw new NotFoundError('File');
    }
    file.is_deleted = false;
    file.deleted_at = null;
    await file.save();
    // Remove from Trash table
    await Trash.destroy({ where: { file_id: file.id, owner_id: req.user.id } });
    logger.info(`File restored from trash: ${file.name} by user ${req.user.email}`);
    res.json({ message: 'File restored from trash', file });
  } catch (error) {
    next(error);
  }
});

// Helper function to recursively hard delete children
async function hardDeleteChildrenRecursively(parentId, ownerId) {
  const children = await File.findAll({
    where: {
      parent_id: parentId,
      owner_id: ownerId
    }
  });
  for (const child of children) {
    // Delete all share links and versions for this file
    await ShareLink.destroy({ where: { file_id: child.id } });
    await FileVersion.destroy({ where: { file_id: child.id } });
    if (child.type === 'folder') {
      await hardDeleteChildrenRecursively(child.id, ownerId);
    }
    await child.destroy({ force: true });
  }
}

// Permanently delete file from trash
router.delete('/:id/permanent', validateUUID('id'), logFileActivity('permanent_delete_file'), async (req, res, next) => {
  try {
    const file = await File.findOne({
      where: {
        id: req.params.id,
        is_deleted: true
      }
    });
    if (!file) {
      // Check if file exists but is not in Trash
      const fileExists = await File.findOne({
        where: {
          id: req.params.id
        }
      });
      if (fileExists) {
        return res.status(400).json({ error: 'File must be in Trash to be permanently deleted.' });
      }
      throw new NotFoundError('File');
    }
    // Delete from S3 if it's a file
    if (file.type === 'file' && file.s3_key) {
      await s3Service.deleteObject(file.s3_key);
    }
    // If it's a folder, recursively delete all S3 objects under its prefix
    if (file.type === 'folder') {
      const userEmail = req.user.email;
      const prefix = `${userEmail}/${file.name}/`;
      let continuationToken = undefined;
      do {
        const response = await s3Service.client.send(
          new ListObjectsV2Command({
            Bucket: s3Service.bucketName,
            Prefix: prefix,
            ContinuationToken: continuationToken
          })
        );
        const objects = (response.Contents || []).map(obj => obj.Key);
        if (objects.length > 0) {
          await s3Service.deleteObjects(objects);
        }
        continuationToken = response.IsTruncated ? response.NextContinuationToken : undefined;
      } while (continuationToken);
      // Recursively hard delete all children in DB
      await hardDeleteChildrenRecursively(file.id, file.owner_id);
    }
    // Delete all share links for this file before deleting the file itself
    await ShareLink.destroy({ where: { file_id: file.id } });
    // Delete all file versions before deleting the file
    await FileVersion.destroy({ where: { file_id: file.id } });
    // Hard delete from DB
    await file.destroy({ force: true });
    // Remove from Trash table
    await Trash.destroy({ where: { file_id: file.id, owner_id: req.user.id } });
    // Recalculate and update user's storage_used
    const user = await User.findByPk(req.user.id);
    const used = await File.sum('size', { where: { owner_id: req.user.id, is_deleted: false, type: 'file' } });
    user.storage_used = used || 0;
    await user.save();
    // Return updated user object in response
    const userObj = user.toJSON();
    userObj.storageUsed = userObj.storage_used;
    userObj.storageQuota = userObj.storage_quota;
    logger.info(`File permanently deleted: ${file.name} by user ${req.user.email}`);
    res.json({ message: 'File permanently deleted', user: userObj });
  } catch (error) {
    next(error);
  }
});

// Sync S3 bucket contents to database for the current user
router.post('/sync-s3', async (req, res, next) => {
  try {
    const userEmail = req.user.email;
    const prefix = `${userEmail}/`;
    let continuationToken = undefined;
    let syncedFiles = [];
    let syncedFolders = [];
    do {
      const response = await s3Service.client.send(
        new ListObjectsV2Command({
          Bucket: s3Service.bucketName,
          Prefix: prefix,
          ContinuationToken: continuationToken
        })
      );
      const contents = response.Contents || [];
      for (const obj of contents) {
        const key = obj.Key;
        if (key.endsWith('/')) {
          // Folder
          const folderName = key.slice(prefix.length, -1);
          if (folderName && !await File.findOne({ where: { name: folderName, owner_id: req.user.id, type: 'folder' } })) {
            const folder = await File.create({
              name: folderName,
              type: 'folder',
              path: `/${folderName}`,
              parent_id: null,
              owner_id: req.user.id
            });
            syncedFolders.push(folder);
          }
        } else {
          // File
          const fileName = key.slice(key.lastIndexOf('/') + 1);
          if (fileName && !await File.findOne({ where: { name: fileName, owner_id: req.user.id, type: 'file' } })) {
            const file = await File.create({
              name: fileName,
              type: 'file',
              path: `/${fileName}`,
              s3_key: key,
              size: obj.Size,
              owner_id: req.user.id
            });
            syncedFiles.push(file);
          }
        }
      }
      continuationToken = response.IsTruncated ? response.NextContinuationToken : undefined;
    } while (continuationToken);

    // Remove DB entries for files/folders that no longer exist in S3 (hard delete, not soft delete)
    const allUserFiles = await File.findAll({ where: { owner_id: req.user.id } });
    const s3Keys = new Set();
    syncedFiles.forEach(f => s3Keys.add(f.s3_key));
    syncedFolders.forEach(f => s3Keys.add(`${userEmail}/${f.name}/`));
    for (const dbFile of allUserFiles) {
      if (dbFile.type === 'file' && !s3Keys.has(dbFile.s3_key)) {
        await ShareLink.destroy({ where: { file_id: dbFile.id } });
        await FileVersion.destroy({ where: { file_id: dbFile.id } });
        await dbFile.destroy({ force: true });
      } else if (dbFile.type === 'folder' && !s3Keys.has(`${userEmail}/${dbFile.name}/`)) {
        await ShareLink.destroy({ where: { file_id: dbFile.id } });
        await hardDeleteChildrenRecursively(dbFile.id, dbFile.owner_id);
        await dbFile.destroy({ force: true });
      }
    }
    res.json({ message: 'S3 sync complete', syncedFiles, syncedFolders });
  } catch (error) {
    next(error);
  }
});

// Helper function to recursively delete children
async function deleteChildrenRecursively(parentId, ownerId) {
  const children = await File.findAll({
    where: {
      parent_id: parentId,
      owner_id: ownerId,
      is_deleted: false
    }
  });

  for (const child of children) {
    await child.softDelete();
    
    if (child.type === 'folder') {
      await deleteChildrenRecursively(child.id, ownerId);
    }
  }
}

// Public share link endpoint (no auth required)
router.get('/share/:token', async (req, res, next) => {
  try {
    const { token } = req.params;
    const shareLink = await ShareLink.findOne({ where: { token } });
    if (!shareLink) return res.status(404).json({ error: 'Invalid or expired share link' });
    if (shareLink.expires_at && new Date() > shareLink.expires_at) {
      return res.status(410).json({ error: 'Share link expired' });
    }
    // Optionally check download limits, password, etc.
    const file = await File.findByPk(shareLink.file_id);
    if (!file) return res.status(404).json({ error: 'File not found' });
    res.json({ file });
  } catch (error) {
    next(error);
  }
});

// Public download endpoint (no auth required)
router.get('/share/:token/download', async (req, res, next) => {
  try {
    const { token } = req.params;
    const shareLink = await ShareLink.findOne({ where: { token } });
    if (!shareLink) return res.status(404).json({ error: 'Invalid or expired share link' });
    if (shareLink.expires_at && new Date() > shareLink.expires_at) {
      return res.status(410).json({ error: 'Share link expired' });
    }
    const file = await File.findByPk(shareLink.file_id);
    if (!file) return res.status(404).json({ error: 'File not found' });
    // Implement your file streaming or S3 download logic here
    // For now, just return file info
    res.json({ file });
  } catch (error) {
    next(error);
  }
});

// Share a file/folder with a user
router.post('/:id/share', async (req, res, next) => {
  try {
    const { userId, permissions } = req.body; // permissions: ['read', 'write']
    if (!userId) {
      return res.status(400).json({ error: 'Missing userId in request body.' });
    }
    // Validate user exists
    const targetUser = await User.findByPk(userId);
    if (!targetUser) {
      return res.status(404).json({ error: 'Target user not found.' });
    }
    const file = await File.findByPk(req.params.id);
    if (!file) return res.status(404).json({ error: 'File not found' });
    // Only owner or admin can share
    if (file.owner_id !== req.user.id && req.user.role !== 'admin') {
      return res.status(403).json({ error: 'Not authorized to share this file/folder' });
    }
    // Prevent duplicate share
    const existingShare = await FileShare.findOne({ where: { file_id: file.id, user_id: userId } });
    if (existingShare) {
      return res.status(409).json({ error: 'File is already shared with this user.' });
    }
    // Create share
    const share = await FileShare.create({
      file_id: file.id,
      user_id: userId,
      permissions: permissions || ['read']
    });
    res.json({ share });
  } catch (error) {
    logger.error('Error in /files/:id/share:', error);
    next(error);
  }
});

// Remove a user's access to a file/folder
router.delete('/:id/share/:userId', async (req, res, next) => {
  try {
    const file = await File.findByPk(req.params.id);
    if (!file) return res.status(404).json({ error: 'File not found' });
    if (file.owner_id !== req.user.id && req.user.role !== 'admin') {
      return res.status(403).json({ error: 'Not authorized to unshare this file/folder' });
    }
    await FileShare.destroy({ where: { file_id: file.id, user_id: req.params.userId } });
    res.json({ message: 'Share removed' });
  } catch (error) {
    next(error);
  }
});

// List all users a file/folder is shared with
router.get('/:id/shares', async (req, res, next) => {
  try {
    const shares = await FileShare.findAll({
      where: { file_id: req.params.id },
      include: [{ model: User, as: 'user', attributes: ['id', 'name', 'email'] }]
    });
    res.json({ shares });
  } catch (error) {
    next(error);
  }
});

// Helper to recursively collect files with their relative paths
async function collectFiles(folderId, basePath = '') {
  const entries = await File.findAll({ where: { parent_id: folderId, is_deleted: false } });
  let files = [];
  for (const entry of entries) {
    if (entry.type === 'file') {
      files.push({ file: entry, path: basePath + entry.name });
    } else if (entry.type === 'folder') {
      files = files.concat(await collectFiles(entry.id, basePath + entry.name + '/'));
    }
  }
  return files;
}

// Download folder as ZIP
router.get('/:id/download-zip', async (req, res, next) => {
  try {
    const folder = await File.findByPk(req.params.id);
    if (!folder || folder.type !== 'folder') {
      return res.status(404).json({ error: 'Folder not found' });
    }
    // Permission: allow if owner, admin, or shared
    const isOwner = folder.owner_id === req.user.id;
    const isAdmin = req.user.role === 'admin';
    let isShared = false;
    if (!isOwner && !isAdmin) {
      const share = await FileShare.findOne({ where: { file_id: folder.id, user_id: req.user.id } });
      isShared = !!share;
    }
    if (!isOwner && !isAdmin && !isShared) {
      return res.status(403).json({ error: 'Not authorized to download this folder' });
    }
    const files = await collectFiles(folder.id);
    res.setHeader('Content-Type', 'application/zip');
    res.setHeader('Content-Disposition', `attachment; filename="${folder.name}.zip"`);
    const archive = archiver('zip', { zlib: { level: 9 } });
    archive.pipe(res);
    for (const { file, path } of files) {
      const s3Stream = await s3Service.getObjectStream(file.s3_key);
      archive.append(s3Stream, { name: path });
    }
    archive.finalize();
  } catch (error) {
    next(error);
  }
});

// Storage usage analytics endpoint
router.get('/usage/analytics', async (req, res, next) => {
  try {
    // Get all non-deleted files for the user
    const files = await File.findAll({
      where: { owner_id: req.user.id, is_deleted: false },
      attributes: ['id', 'name', 'type', 'size', 'mime_type', 'parent_id', 'created_at', 'updated_at']
    });
    // Aggregate by type, mime, and folder
    let total = 0;
    let byType = {};
    let byMime = {};
    let byFolder = {};
    for (const file of files) {
      if (file.type === 'file') {
        total += file.size || 0;
        // By type
        byType[file.type] = (byType[file.type] || 0) + (file.size || 0);
        // By mime
        if (file.mime_type) byMime[file.mime_type] = (byMime[file.mime_type] || 0) + (file.size || 0);
        // By folder
        const folderId = file.parent_id || 'root';
        byFolder[folderId] = (byFolder[folderId] || 0) + (file.size || 0);
      } else {
        // Ensure folder type/folderId keys exist, but do not add size
        byType[file.type] = byType[file.type] || 0;
        const folderId = file.parent_id || 'root';
        byFolder[folderId] = byFolder[folderId] || 0;
      }
    }
    res.json({
      total,
      byType,
      byMime,
      byFolder,
      files
    });
  } catch (error) {
    next(error);
  }
});

// Debug endpoint: return raw file list for current user
router.get('/usage/files-debug', async (req, res, next) => {
  try {
    const files = await File.findAll({
      where: { owner_id: req.user.id, is_deleted: false },
      order: [['created_at', 'ASC']]
    });
    res.json({ files });
  } catch (error) {
    next(error);
  }
});

export default router;