import { 
  S3Client, 
  PutObjectCommand, 
  GetObjectCommand, 
  DeleteObjectCommand,
  HeadObjectCommand,
  ListObjectsV2Command,
  CopyObjectCommand,
  DeleteObjectsCommand // <-- add this
} from '@aws-sdk/client-s3';
import { getSignedUrl } from '@aws-sdk/s3-request-presigner';
import { s3Client, S3_CONFIG } from '../config/aws.js';
import { logger } from './logger.js';
import crypto from 'crypto';

export class S3Service {
  constructor() {
    this.client = s3Client;
    this.bucketName = S3_CONFIG.bucketName;
  }

  // Generate S3 key for file (now uses user email as root folder)
  generateFileKey(userEmail, fileId, fileName) {
    return `${userEmail}/${fileId}/${fileName}`;
  }

  // Generate S3 key for folder (for zero-byte S3 folder objects)
  generateFolderKey(userEmail, folderPath) {
    // folderPath should be like 'folder1/folder2/'
    return `${userEmail}/${folderPath.endsWith('/') ? folderPath : folderPath + '/'}`;
  }

  // Generate S3 key for file version
  generateVersionKey(userId, fileId, version, fileName) {
    return `users/${userId}/files/${fileId}/versions/${version}/${fileName}`;
  }

  // Generate S3 key for thumbnail
  generateThumbnailKey(userId, fileId, fileName) {
    const ext = fileName.split('.').pop();
    const baseName = fileName.replace(`.${ext}`, '');
    return `users/${userId}/thumbnails/${fileId}/${baseName}_thumb.jpg`;
  }

  // Get presigned URL for upload
  async getUploadUrl(key, contentType, expiresIn = S3_CONFIG.presignedUrlExpiry) {
    try {
      const command = new PutObjectCommand({
        Bucket: this.bucketName,
        Key: key,
        ContentType: contentType
        // Removed ServerSideEncryption: 'AES256' to avoid S3 403 error
      });

      const url = await getSignedUrl(this.client, command, { expiresIn });
      
      logger.info(`Generated upload URL for key: ${key}`);
      return url;
    } catch (error) {
      logger.error('Error generating upload URL:', error);
      throw error;
    }
  }

  // Get presigned URL for download
  async getDownloadUrl(key, expiresIn = S3_CONFIG.presignedUrlExpiry) {
    try {
      const command = new GetObjectCommand({
        Bucket: this.bucketName,
        Key: key
      });

      const url = await getSignedUrl(this.client, command, { expiresIn });
      
      logger.info(`Generated download URL for key: ${key}`);
      return url;
    } catch (error) {
      logger.error('Error generating download URL:', error);
      throw error;
    }
  }

  // Check if object exists
  async objectExists(key) {
    try {
      await this.client.send(new HeadObjectCommand({
        Bucket: this.bucketName,
        Key: key
      }));
      return true;
    } catch (error) {
      if (error.name === 'NotFound') {
        return false;
      }
      throw error;
    }
  }

  // Get object metadata
  async getObjectMetadata(key) {
    try {
      const command = new HeadObjectCommand({
        Bucket: this.bucketName,
        Key: key
      });

      const response = await this.client.send(command);
      
      return {
        size: response.ContentLength,
        lastModified: response.LastModified,
        contentType: response.ContentType,
        etag: response.ETag,
        versionId: response.VersionId
      };
    } catch (error) {
      logger.error('Error getting object metadata:', error);
      throw error;
    }
  }

  // Delete object
  async deleteObject(key, versionId = null) {
    try {
      const command = new DeleteObjectCommand({
        Bucket: this.bucketName,
        Key: key,
        ...(versionId && { VersionId: versionId })
      });

      await this.client.send(command);
      logger.info(`Deleted object: ${key}${versionId ? ` (version: ${versionId})` : ''}`);
    } catch (error) {
      logger.error('Error deleting object:', error);
      throw error;
    }
  }

  // Copy object
  async copyObject(sourceKey, destinationKey) {
    try {
      const command = new CopyObjectCommand({
        Bucket: this.bucketName,
        CopySource: `${this.bucketName}/${sourceKey}`,
        Key: destinationKey,
        ServerSideEncryption: 'AES256'
      });

      const response = await this.client.send(command);
      logger.info(`Copied object from ${sourceKey} to ${destinationKey}`);
      
      return {
        etag: response.CopyObjectResult.ETag,
        lastModified: response.CopyObjectResult.LastModified
      };
    } catch (error) {
      logger.error('Error copying object:', error);
      throw error;
    }
  }

  // List objects with prefix
  async listObjects(prefix, maxKeys = 1000) {
    try {
      const command = new ListObjectsV2Command({
        Bucket: this.bucketName,
        Prefix: prefix,
        MaxKeys: maxKeys
      });

      const response = await this.client.send(command);
      
      return {
        objects: response.Contents || [],
        isTruncated: response.IsTruncated,
        nextContinuationToken: response.NextContinuationToken
      };
    } catch (error) {
      logger.error('Error listing objects:', error);
      throw error;
    }
  }

  // Batch delete objects (for folder recursive delete)
  async deleteObjects(keys = []) {
    if (!keys.length) return;
    try {
      const command = new DeleteObjectsCommand({
        Bucket: this.bucketName,
        Delete: {
          Objects: keys.map(key => ({ Key: key }))
        }
      });
      await this.client.send(command);
      logger.info(`Batch deleted objects: ${keys.join(', ')}`);
    } catch (error) {
      logger.error('Error batch deleting objects:', error);
      throw error;
    }
  }

  // Generate file checksum
  generateChecksum(buffer) {
    return crypto.createHash('md5').update(buffer).digest('hex');
  }

  // Validate file type
  isAllowedFileType(mimeType) {
    if (S3_CONFIG.allowedFileTypes.includes('*')) {
      return true;
    }
    
    return S3_CONFIG.allowedFileTypes.some(allowedType => {
      if (allowedType.endsWith('/*')) {
        const category = allowedType.replace('/*', '');
        return mimeType.startsWith(category + '/');
      }
      return mimeType === allowedType;
    });
  }

  // Validate file size
  isAllowedFileSize(size) {
    return size <= S3_CONFIG.maxFileSize;
  }

  // Get file category from MIME type
  getFileCategory(mimeType) {
    if (!mimeType) return 'other';
    
    if (mimeType.startsWith('image/')) return 'image';
    if (mimeType.startsWith('video/')) return 'video';
    if (mimeType.startsWith('audio/')) return 'audio';
    if (mimeType.includes('pdf')) return 'document';
    if (mimeType.includes('text/') || mimeType.includes('document')) return 'document';
    if (mimeType.includes('zip') || mimeType.includes('rar') || mimeType.includes('tar')) return 'archive';
    
    return 'other';
  }

  // Get S3 object as stream (for direct preview)
  async getObjectStream(key) {
    const command = new GetObjectCommand({
      Bucket: this.bucketName,
      Key: key
    });
    const response = await this.client.send(command);
    return response.Body;
  }
}

export const s3Service = new S3Service();