import { DataTypes } from 'sequelize';
import { sequelize } from '../config/database.js';

export const UserSession = sequelize.define('user_sessions', {
  id: {
    type: DataTypes.UUID,
    defaultValue: DataTypes.UUIDV4,
    primaryKey: true
  },
  user_id: {
    type: DataTypes.UUID,
    allowNull: false,
    references: {
      model: 'users',
      key: 'id'
    }
  },
  token: {
    type: DataTypes.TEXT, // Allow long JWT tokens
    allowNull: false,
    unique: true
  },
  ip_address: {
    type: DataTypes.INET,
    allowNull: true
  },
  user_agent: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  expires_at: {
    type: DataTypes.DATE,
    allowNull: false
  },
  is_active: {
    type: DataTypes.BOOLEAN,
    defaultValue: true
  },
  last_activity: {
    type: DataTypes.DATE,
    defaultValue: DataTypes.NOW
  }
}, {
  indexes: [
    {
      fields: ['user_id']
    },
    {
      fields: ['token']
    },
    {
      fields: ['is_active']
    },
    {
      fields: ['expires_at']
    }
  ]
});

// Instance methods
UserSession.prototype.isExpired = function() {
  return new Date() > this.expires_at;
};

UserSession.prototype.updateActivity = async function() {
  this.last_activity = new Date();
  await this.save();
};

UserSession.prototype.revoke = async function() {
  this.is_active = false;
  await this.save();
};

// Class methods
UserSession.findByToken = function(token) {
  return this.findOne({
    where: {
      token,
      is_active: true
    },
    include: ['user']
  });
};

UserSession.findByUser = function(userId) {
  return this.findAll({
    where: {
      user_id: userId,
      is_active: true
    },
    order: [['last_activity', 'DESC']]
  });
};

UserSession.cleanupExpired = async function() {
  const expiredCount = await this.update(
    { is_active: false },
    {
      where: {
        expires_at: {
          [sequelize.Sequelize.Op.lt]: new Date()
        },
        is_active: true
      }
    }
  );
  
  return expiredCount;
};

UserSession.revokeAllForUser = async function(userId, exceptToken = null) {
  const where = {
    user_id: userId,
    is_active: true
  };
  
  if (exceptToken) {
    where.token = {
      [sequelize.Sequelize.Op.ne]: exceptToken
    };
  }
  
  return this.update(
    { is_active: false },
    { where }
  );
};