import { S3Client } from '@aws-sdk/client-s3';
import { logger } from '../utils/logger.js';

// Validate AWS configuration
const requiredEnvVars = ['AWS_REGION', 'AWS_ACCESS_KEY_ID', 'AWS_SECRET_ACCESS_KEY', 'S3_BUCKET_NAME'];
const missingVars = requiredEnvVars.filter(varName => !process.env[varName]);

if (missingVars.length > 0) {
  logger.error(`Missing required AWS environment variables: ${missingVars.join(', ')}`);
  if (process.env.NODE_ENV === 'production') {
    process.exit(1);
  }
}

// Create S3 client
export const s3Client = new S3Client({
  region: process.env.AWS_REGION || 'us-east-1',
  credentials: {
    accessKeyId: process.env.AWS_ACCESS_KEY_ID || '',
    secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY || ''
  }
});

export const S3_CONFIG = {
  bucketName: process.env.S3_BUCKET_NAME || '',
  region: process.env.S3_BUCKET_REGION || process.env.AWS_REGION || 'us-east-1',
  maxFileSize: parseInt(process.env.MAX_FILE_SIZE) || 100 * 1024 * 1024, // 100MB
  allowedFileTypes: process.env.ALLOWED_FILE_TYPES?.split(',') || ['*'],
  presignedUrlExpiry: 3600 // 1 hour
};

logger.info(`AWS S3 configured for region: ${S3_CONFIG.region}, bucket: ${S3_CONFIG.bucketName}`);