import express from 'express';
import jwt from 'jsonwebtoken';
import { User, UserSession, File } from '../models/index.js';
import { validateLogin, validateRegister } from '../middleware/validation.js';
import { logAuthActivity } from '../middleware/logging.js';
import { AppError } from '../middleware/errorHandler.js';
import { logger } from '../utils/logger.js';
import { s3Service } from '../utils/s3.js';
import { v4 as uuidv4 } from 'uuid';
import { PutObjectCommand } from '@aws-sdk/client-s3';
import rateLimit from 'express-rate-limit';

const router = express.Router();

// Helper to extract username from email
function getUsernameFromEmail(email) {
  return email.split('@')[0];
}

// Rate limiter for login route
const loginLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 10, // limit each IP to 10 login requests per windowMs
  message: 'Too many login attempts from this IP, please try again later.'
});

// Register new user
router.post('/register', validateRegister, logAuthActivity('register'), async (req, res, next) => {
  try {
    const { email, password, name } = req.body;

    // Check if user already exists
    const existingUser = await User.findByEmail(email);
    if (existingUser) {
      throw new AppError('User already exists with this email', 409);
    }

    // Create new user
    const user = await User.create({
      email: email.toLowerCase(),
      password,
      name,
      is_verified: true // Auto-verify for demo
    });

    // Create user root folder in S3 and DB
    const username = getUsernameFromEmail(user.email);
    const s3RootKey = `${username}/`;
    // S3: create a zero-byte object to represent the folder
    await s3Service.client.send(new PutObjectCommand({
      Bucket: s3Service.bucketName,
      Key: s3RootKey
    }));
    // DB: create root folder record
    await File.create({
      name: username,
      type: 'folder',
      size: 0,
      mime_type: null,
      path: '/',
      s3_key: s3RootKey,
      parent_id: null,
      owner_id: user.id,
      is_shared: false,
      is_starred: false,
      is_deleted: false,
      deleted_at: null,
      permissions: ['read', 'write', 'delete'],
      metadata: { root: true }
    });

    // Generate JWT token
    const token = jwt.sign(
      { userId: user.id, email: user.email, role: user.role },
      process.env.JWT_SECRET,
      { expiresIn: process.env.JWT_EXPIRES_IN || '7d' }
    );

    // Create user session
    const expiresAt = new Date();
    expiresAt.setDate(expiresAt.getDate() + 7); // 7 days

    await UserSession.create({
      user_id: user.id,
      token,
      ip_address: req.ip,
      user_agent: req.get('User-Agent'),
      expires_at: expiresAt
    });

    logger.info(`User registered: ${user.email}`);

    res.status(201).json({
      message: 'User registered successfully',
      user: user.toJSON(),
      token
    });
  } catch (error) {
    next(error);
  }
});

// Login user
router.post('/login', loginLimiter, validateLogin, logAuthActivity('login'), async (req, res, next) => {
  try {
    const { email, password } = req.body;
    logger.debug(`[LOGIN] RAW email received: '${req.body.email}'`);
    logger.debug(`[LOGIN] Attempt: email=${email}, password_length=${password ? password.length : 0}`);

    // Find user by email (no forced connection close)
    const user = await User.findOne({ 
      where: { email: email.toLowerCase() },
      attributes: { include: ['password', 'login_attempts', 'lock_until'] }
    });
    if (user) {
      await user.reload(); // Force reload from DB
    }
    logger.debug(`[LOGIN] User found: ${!!user}, is_active: ${user ? user.is_active : 'N/A'}`);
    if (user) {
      logger.debug(`[LOGIN] User password hash: ${user.password}`);
    }

    // Check if account is locked
    if (user && user.lock_until && user.lock_until > new Date()) {
      logger.warn(`[LOGIN] Account locked for email=${email} until ${user.lock_until}`);
      throw new AppError('Account is temporarily locked due to too many failed login attempts. Please try again later.', 423);
    }

    if (!user || !user.is_active) {
      logger.warn(`[LOGIN] Invalid credentials: user not found or inactive for email=${email}`);
      throw new AppError('Invalid credentials', 401);
    }

    // Check password
    const isPasswordValid = await user.comparePassword(password);
    logger.debug(`[LOGIN] comparePassword result: ${isPasswordValid}`);
    if (!isPasswordValid) {
      // Increment login attempts
      user.login_attempts = (user.login_attempts || 0) + 1;
      // Lock account for 15 min after 5 failed attempts
      if (user.login_attempts >= 5) {
        user.lock_until = new Date(Date.now() + 15 * 60 * 1000);
        await user.save();
        logger.warn(`[LOGIN] Account locked for email=${email} after too many failed attempts.`);
        throw new AppError('Account is temporarily locked due to too many failed login attempts. Please try again later.', 423);
      } else {
        await user.save();
      }
      logger.warn(`[LOGIN] Invalid credentials: password mismatch for email=${email}`);
      throw new AppError('Invalid credentials', 401);
    }

    // Reset login attempts on successful login
    user.login_attempts = 0;
    user.lock_until = null;
    user.last_login = new Date();
    await user.save();

    // Generate JWT token
    const token = jwt.sign(
      { userId: user.id, email: user.email, role: user.role },
      process.env.JWT_SECRET,
      { expiresIn: process.env.JWT_EXPIRES_IN || '7d' }
    );

    // Create user session
    const expiresAt = new Date();
    expiresAt.setDate(expiresAt.getDate() + 7); // 7 days

    await UserSession.create({
      user_id: user.id,
      token,
      ip_address: req.ip,
      user_agent: req.get('User-Agent'),
      expires_at: expiresAt
    });

    logger.info(`User logged in: ${user.email}`);

    res.json({
      message: 'Login successful',
      user: user.toJSON(),
      token
    });
  } catch (error) {
    next(error);
  }
});

// Logout user
router.post('/logout', async (req, res, next) => {
  try {
    const authHeader = req.headers.authorization;
    
    if (authHeader && authHeader.startsWith('Bearer ')) {
      const token = authHeader.substring(7);
      
      // Find and revoke session
      const session = await UserSession.findByToken(token);
      if (session) {
        await session.revoke();
        logger.info(`User logged out: ${session.user?.email}`);
      }
    }

    res.json({ message: 'Logout successful' });
  } catch (error) {
    next(error);
  }
});

// Refresh token
router.post('/refresh', async (req, res, next) => {
  try {
    const authHeader = req.headers.authorization;
    
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
      throw new AppError('Access token required', 401);
    }

    const token = authHeader.substring(7);
    
    // Verify current token
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    
    // Find session
    const session = await UserSession.findByToken(token);
    if (!session || session.isExpired()) {
      throw new AppError('Invalid or expired session', 401);
    }

    // Get user
    const user = await User.findByPk(decoded.userId);
    if (!user || !user.is_active) {
      throw new AppError('User not found or inactive', 401);
    }

    // Generate new token
    const newToken = jwt.sign(
      { userId: user.id, email: user.email, role: user.role },
      process.env.JWT_SECRET,
      { expiresIn: process.env.JWT_EXPIRES_IN || '7d' }
    );

    // Update session
    session.token = newToken;
    session.expires_at = new Date(Date.now() + 7 * 24 * 60 * 60 * 1000); // 7 days
    await session.save();

    res.json({
      message: 'Token refreshed successfully',
      token: newToken
    });
  } catch (error) {
    next(error);
  }
});

// Get current user
router.get('/me', async (req, res, next) => {
  try {
    const authHeader = req.headers.authorization;
    
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
      throw new AppError('Access token required', 401);
    }

    const token = authHeader.substring(7);
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    
    const user = await User.findByPk(decoded.userId);
    if (!user || !user.is_active) {
      throw new AppError('User not found or inactive', 401);
    }

    // Always recalculate storage_used before returning user info
    const used = await File.sum('size', { where: { owner_id: user.id, is_deleted: false, type: 'file' } });
    if (user.storage_used !== (used || 0)) {
      // Only update if changed, and avoid triggering hooks
      await user.update({ storage_used: used || 0 }, { hooks: false });
      user.storage_used = used || 0;
    }
    // Add camelCase fields for frontend compatibility
    const userObj = user.toJSON();
    userObj.storageUsed = userObj.storage_used;
    userObj.storageQuota = userObj.storage_quota;

    res.json({ user: userObj });
  } catch (error) {
    next(error);
  }
});

export default router;